/// <reference path="v4PrintDriver-Intellisense.js" />

// --------------------------------------------------------------------------
// Note: To disable intellisense for Windows 8.1 APIs, please delete the line below
/// <reference path="v4PrintDriver-Intellisense-Windows8.1.js" />
// --------------------------------------------------------------------------

var pskNs = "http://schemas.microsoft.com/windows/2003/08/printing/printschemakeywords";
var pskV11Ns = "http://schemas.microsoft.com/windows/2013/05/printing/printschemakeywordsv11";
var psfNs = "http://schemas.microsoft.com/windows/2003/08/printing/printschemaframework";
var ns0000Ns = "http://schemas.brother.info/mfc/printing/2006/11/printschemakeywords"

var psfPrefix = "psf";

function validatePrintTicket(printTicket, scriptContext) {
    /// <param name="printTicket" type="IPrintSchemaTicket">
    ///     Print ticket to be validated.
    /// </param>
    /// <param name="scriptContext" type="IPrinterScriptContext">
    ///     Script context object.
    /// </param>
    /// <returns type="Number" integer="true">
    ///     Integer value indicating validation status.
    ///         1 - Print ticket is valid and was not modified.
    ///         2 - Print ticket was modified to make it valid.
    ///         0 - Print ticket is invalid.
    /// </returns>

    var retVal = 1;

    setSelectionNamespace(printTicket.XmlNode, psfPrefix, psfNs);

    if (isPageResolutionConstrainedByPageOutputQuality(printTicket)) {
        var printTicketPageResolutionFeature = getPrintTicketPageResolutionFeature(printTicket);
        var printCapabilitiesPageResolutionFeature = getPrintCapabilitiesPageResolutionFeature(printTicket);

        var allowedPageResolutionOption =
            printCapabilitiesPageResolutionFeature.GetOption(pageResolutionOptions.getAllowedPageResolutionOptionName(getSelectedPageOutputQualityOptionName(printTicket)), ns0000Ns);

        printTicketPageResolutionFeature.SelectedOption = allowedPageResolutionOption;

        retVal = 2;
    }

    // Below demonstrates correct usage of IPrintSchemaTicket2 APIs so that the script does not terminate
    // when running on a Windows 8 version of PrintConfig.dll.
    if (printSchemaApiHelpers.supportsIPrintSchemaTicket2(printTicket)) {
        var param = printTicket.GetParameterInitializer("JobCopiesAllDocuments");
    }

    return retVal;
}

var pageResolutionOptions = {

    allowedPageResolutionOptionForDraft: "Option1",
    allowedPageResolutionOptionForNormal: "Option2",
    allowedPageResolutionOptionForHigh: "Option3",

    getAllowedPageResolutionOptionName: function (pageOutputQualityOptionName) {
        switch (pageOutputQualityOptionName) {
            case "Draft":
                return pageResolutionOptions.allowedPageResolutionOptionForDraft;
                break;
            case "Normal":
                return pageResolutionOptions.allowedPageResolutionOptionForNormal;
                break;
            case "High":
                return pageResolutionOptions.allowedPageResolutionOptionForHigh;
                break;
            default:
                return null;
                break;
        }
    }
}

function getPrintTicketPageResolutionFeature(printTicket) {
    return printTicket.GetFeature("PageResolution");
}

function getPrintCapabilitiesPageResolutionFeature(printTicket) {
    var printCapabilities = printTicket.GetCapabilities();
    return printCapabilities.GetFeature("PageResolution");
}

function getSelectedPageResolutionOptionName (printTicket) {
    var pageResolutionFeature = getPrintTicketPageResolutionFeature(printTicket);

    if (pageResolutionFeature) {
        var pageResolutionOptionNamespaceUri = pageResolutionFeature.SelectedOption.NamespaceUri;
        var pageResolutionOptionName = pageResolutionFeature.SelectedOption.Name;

        if (pageResolutionOptionNamespaceUri == ns0000Ns) {
            return pageResolutionOptionName;
        }
    }
    return null;
}

function getSelectedPageOutputQualityOptionName (printTicket) {
    var pageOutputQualityFeature = printTicket.GetFeature("PageOutputQuality");

    if (pageOutputQualityFeature) {
        var pageOutputQualityOptionNamespaceUri = pageOutputQualityFeature.SelectedOption.NamespaceUri;
        var pageOutputQualityOptionName = pageOutputQualityFeature.SelectedOption.Name;

        if (pageOutputQualityOptionNamespaceUri == pskNs) {
            return pageOutputQualityOptionName;
        }
    }
    return null;
}

function isPageResolutionConstrainedByPageOutputQuality(printTicket) {
    var selectedPageOutputQualityOptionName = getSelectedPageOutputQualityOptionName(printTicket);
    var selectedPageResolutionOptionName = getSelectedPageResolutionOptionName(printTicket);

    if (selectedPageOutputQualityOptionName && selectedPageResolutionOptionName) {
        var allowedPageResolutionOptionName = pageResolutionOptions.getAllowedPageResolutionOptionName(selectedPageOutputQualityOptionName);

        if (allowedPageResolutionOptionName) {
            if (allowedPageResolutionOptionName !== selectedPageResolutionOptionName) {
                return true;
            }
        }
    }
    return false;
}

function setSelectionNamespace(xmlNode, prefix, namespace) {
    /// <summary>
    ///     This function sets the 'SelectionNamespaces' property on the XML Node.
    ///     For more details: http://msdn.microsoft.com/en-us/library/ms756048(VS.85).aspx
    /// </summary>
    /// <param name="xmlNode" type="IXMLDOMNode">
    ///     The node on which the property is set.
    /// </param>
    /// <param name="prefix" type="String">
    ///     The prefix to be associated with the namespace.
    /// </param>
    /// <param name="namespace" type="String">
    ///     The namespace to be added to SelectionNamespaces.
    /// </param>

    xmlNode.setProperty("SelectionNamespaces", "xmlns:" + prefix + "='" + namespace + "'");
}

function completePrintCapabilities(printTicket, scriptContext, printCapabilities) {
    /// <param name="printTicket" type="IPrintSchemaTicket" mayBeNull="true">
    ///     If not 'null', the print ticket's settings are used to customize the print capabilities.
    /// </param>
    /// <param name="scriptContext" type="IPrinterScriptContext">
    ///     Script context object.
    /// </param>
    /// <param name="printCapabilities" type="IPrintSchemaCapabilities">
    ///     Print capabilities object to be customized.
    /// </param>

    // Below demonstrates correct usage of IPrintSchemaCapabilities2 APIs so that the script does not terminate
    // when running on a Windows 8 version of PrintConfig.dll.
    if (printSchemaApiHelpers.supportsIPrintSchemaCapabilities2(printCapabilities)) {
        var param = printCapabilities.GetParameterDefinition("JobCopiesAllDocuments");
    }
}

function convertPrintTicketToDevMode(printTicket, scriptContext, devModeProperties) {
    /// <param name="printTicket" type="IPrintSchemaTicket">
    ///     Print ticket to be converted to DevMode.
    /// </param>
    /// <param name="scriptContext" type="IPrinterScriptContext">
    ///     Script context object.
    /// </param>
    /// <param name="devModeProperties" type="IPrinterScriptablePropertyBag">
    ///     The DevMode property bag.
    /// </param>
}


function convertDevModeToPrintTicket(devModeProperties, scriptContext, printTicket) {
    /// <param name="devModeProperties" type="IPrinterScriptablePropertyBag">
    ///     The DevMode property bag.
    /// </param>
    /// <param name="scriptContext" type="IPrinterScriptContext">
    ///     Script context object.
    /// </param>
    /// <param name="printTicket" type="IPrintSchemaTicket">
    ///     Print ticket to be converted from the DevMode.
    /// </param>
}

var PrintSchemaConstrainedSetting = {
    PrintSchemaConstrainedSetting_None: 0,
    PrintSchemaConstrainedSetting_PrintTicket: 1,
    PrintSchemaConstrainedSetting_Admin: 2,
    PrintSchemaConstrainedSetting_Device: 3
};

var PrintSchemaParameterDataType = {
    PrintSchemaParameterDataType_Integer: 0,
    PrintSchemaParameterDataType_NumericString: 1,
    PrintSchemaParameterDataType_String: 2
};

var STREAM_SEEK = {
    STREAM_SEEK_SET: 0,
    STREAM_SEEK_CUR: 1,
    STREAM_SEEK_END: 2
};

var PrintSchemaSelectionType = {
    PrintSchemaSelectionType_PickOne: 0,
    PrintSchemaSelectionType_PickMany: 1
};


var printSchemaApiHelpers = {
    supportsIPrintSchemaCapabilities2: function (printCapabilities) {
        /// <summary>
        ///     Determines if the IPrintSchemaCapabilities2 APIs are supported on the 'printCapabilities' object.
        /// </summary>
        /// <param name="printCapabilities" type="IPrintSchemaCapabilities">
        ///     Print capabilities object.
        /// </param>
        /// <returns type="Boolean">
        ///     true - the interface APIs are supported.
        ///     false - the interface APIs are not supported.
        /// </returns>

        var supported = true;

        try {
            if (typeof printCapabilities.getParameterDefinition === "undefined") {
                supported = false;
            }
        }
        catch (exception) {
            supported = false;
        }

        return supported;
    },
    supportsIPrintSchemaTicket2: function (printTicket) {
        /// <summary>
        ///     Determines if the IPrintSchemaTicket2 APIs are supported on the 'printTicket' object.
        /// </summary>
        /// <param name="printTicket" type="IPrintSchemaTicket">
        ///     Print ticket object.
        /// </param>
        /// <returns type="Boolean">
        ///     true - the interface APIs are supported.
        ///     false - the interface APIs are not supported.
        /// </returns>

        var supported = true;

        try {
            if (typeof printTicket.getParameterInitializer === "undefined") {
                supported = false;
            }
        }
        catch (exception) {
            supported = false;
        }

        return supported;
    }
}